package com.agilex.healthcare.directscheduling.facilityProvider;

import static org.mockito.Mockito.*;

import java.util.*;

import com.agilex.healthcare.directscheduling.dao.DAOConstants;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.runners.MockitoJUnitRunner;

import com.agilex.healthcare.directscheduling.dao.PatientRelationshipDaoImpl;
import com.agilex.healthcare.directscheduling.dataservice.DSFacilityProviderDataService;
import com.agilex.healthcare.directscheduling.domain.DirectSchedulingFacilityProvider;
import com.agilex.healthcare.directscheduling.domain.DirectSchedulingFacilityProviders;
import com.agilex.healthcare.directscheduling.domain.FacilityMember;
import com.agilex.healthcare.directscheduling.domain.FacilityMemberClinic;
import com.agilex.healthcare.directscheduling.domain.FacilityMemberProvider;
import com.agilex.healthcare.directscheduling.domain.FacilityMemberTeam;
import com.agilex.healthcare.directscheduling.domain.VARFacilityMembers;
import com.agilex.healthcare.directscheduling.domain.facility.DSFacilities;
import com.agilex.healthcare.directscheduling.domain.facility.DSFacility;
import com.agilex.healthcare.directscheduling.domain.facility.DSFacilityClinic;
import com.agilex.healthcare.directscheduling.domain.facility.DSFacilityProvider;
import com.agilex.healthcare.directscheduling.resource.DSResource;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.veteranappointment.utils.SpringBasedIntegrationRepo;
import gov.va.vamf.scheduling.direct.datalayer.user.VarPatientCorrelationGateway;

@RunWith(MockitoJUnitRunner.class)
public class DSFacilityProviderDataServiceTest extends SpringBasedIntegrationRepo {

	private static final String ASSIGNING_AUTH_EDIPI = "EDIPI";
	private static final String ASSIGNING_AUTH_ICN = "ICN";
	private static final String ASSIGNING_AUTH_DFN = "dfn-777" ;
	private static final String ASSIGNING_AUTH_MH = "MH" ;

	private static final String PATIENT_ID_EDIPI = "P001";
	private static final String PATIENT_ID_ICN = "1008593069V747998";
	private static final String PATIENT_ID_DFN = "237" ;
	private static final String PATIENT_ID_MH = "1011214432V662527" ;
	private static final String STOP_CODE_PAIRS = "[323]" ;

	@InjectMocks
	private DSFacilityProviderDataService dsFacilityProviderDataService;

	@Mock
	private PatientRelationshipDaoImpl patientRelationshipDaoTeamImpl;
	@Mock
	private PatientRelationshipDaoImpl patientRelationshipDaoClinicImpl;
	@Mock
	private VarPatientCorrelationGateway patientCorrelationGateway;

	@Mock
	private DSResource dsResource;

	@Before
	public void init() {

		MockitoAnnotations.initMocks(this);
	}

	@Test
	public void getPactFacilityMembersForPatientTest( ) {

		// Setup data
		PatientIdentifier mockPatientIdentifierEDIPI = new PatientIdentifier( ASSIGNING_AUTH_EDIPI , PATIENT_ID_EDIPI ) ;

		setupMockPatientCorrespondIdentifiers( mockPatientIdentifierEDIPI ) ;
		setupMockVarFacilityTeamMembers( ) ;
		setupMockVarFacilityClinicMembers( ) ;

		// Test call
		DirectSchedulingFacilityProviders dsfps = dsFacilityProviderDataService
			.getPactFacilityMembersForPatient(mockPatientIdentifierEDIPI, true, "[323]") ;
		assertNotNull( dsfps ) ;

		List< DirectSchedulingFacilityProvider > dsfpsList = dsfps.getFacilityProviderList( ) ;
		assertNotNull( dsfpsList ) ;
		assertTrue( dsfpsList.size( ) > 0 ) ;

		for( DirectSchedulingFacilityProvider dsfp : dsfpsList ) {

			assertNotNull( dsfp ) ;
			assertNotNull( dsfp.getFacilityId( ) ) ;
			assertNotNull( dsfp.getInstitutionName( ) ) ;

			Collection< FacilityMemberTeam > teamCollection = dsfp.getTeamList() ;
			assertTrue( teamCollection != null && teamCollection.size( ) > 0 ) ;
    		for( FacilityMemberTeam teamMember : teamCollection ) {

    			assertNotNull( teamMember.getStaffName( ) ) ;
    			assertNotNull( teamMember.getProviderStandardPosition( ) ) ;

    			assertNotNull( teamMember.getFacilityId() ) ;
    			assertNotNull( teamMember.getPossiblePrimary()) ;
    			assertNotNull( teamMember.getProviderPosition());
    			assertNotNull( teamMember.getProviderSid()) ;
    			assertNotNull( teamMember.getTeamName() ) ;
    			assertNotNull( teamMember.getTeamPurpose()) ;
    			assertNotNull( teamMember.getTeamSid()) ;
    		}

			Collection< FacilityMemberClinic > clinicCollection = dsfp.getClinicList( ) ;
			assertTrue( clinicCollection != null && clinicCollection.size( ) > 0 ) ;
    		for( FacilityMemberClinic clinic : clinicCollection ) {

    			assertNotNull( clinic.getLocationIEN( ) ) ;
    			assertNotNull( clinic.getPhysicalLocation( ) ) ;

    			assertNotNull( clinic.getFacilityId( ) ) ;
    			assertNotNull( clinic.getInstitutionCode() ) ;
    			assertNotNull( clinic.getInstitutionIEN()) ;
    			assertNotNull( clinic.getInstitutionName()) ;
    			assertNotNull( clinic.getInstitutionSID()) ;
    			assertNotNull( clinic.getLocationName()) ;
    			assertNotNull( clinic.getLocationSID()) ;
    			assertNotNull( clinic.getLocationType()) ;
    			assertNotNull( clinic.getPrimaryStopCode()) ;
    			assertNotNull( clinic.getPrimaryStopCodeName()) ;
    			assertNotNull( clinic.getTeamName() ) ;
    			assertNotNull( clinic.getTeamPurpose() ) ;
    			assertNotNull( clinic.getTeamSID() ) ;
    		}

			Collection< FacilityMemberTeam > primaryCollection = dsfp.getPrimaryList() ;
			assertTrue( primaryCollection != null && primaryCollection.size() > 0 ) ;
			for( FacilityMemberTeam primary : primaryCollection ) {

    			assertNotNull( primary.getStaffName() ) ;
    			assertNotNull( primary.getProviderStandardPosition()) ;

    			assertNotNull( primary.getFacilityId()) ;
    			assertNotNull( primary.getPossiblePrimary()) ;
    			assertNotNull( primary.getProviderPosition()) ;
    			assertNotNull( primary.getProviderSid()) ;
    			assertNotNull( primary.getTeamName()) ;
    			assertNotNull( primary.getTeamPurpose()) ;
    			assertNotNull( primary.getTeamSid()) ;
			}
		}
	}

	@Test
	public void getAllFacilityProvidersForPatientTest() {

		// Setup data
		PatientIdentifier mockPatientIdentifierEDIPI = new PatientIdentifier( ASSIGNING_AUTH_EDIPI , PATIENT_ID_EDIPI);

		setupMockPatientCorrespondIdentifiers( mockPatientIdentifierEDIPI ) ;

		// Test DataService call
		DSFacilities dsFacilities = dsFacilityProviderDataService.getAllFacilityProvidersForPatient(mockPatientIdentifierEDIPI);
		assertNotNull(dsFacilities);
		assertNotNull(dsFacilities.getFacilityList());

		assertTrue(dsFacilities.getFacilityList().size() > 0);
		for (DSFacility dsFacility : dsFacilities.getFacilityList()) {

			assertNotNull(dsFacility);

			assertNotNull(dsFacility.getFacilityAddr1());
			assertNotNull(dsFacility.getFacilityAddr2());
			assertNotNull(dsFacility.getFacilityCity());
			assertNotNull(dsFacility.getFacilityCode());
			assertNotNull(dsFacility.getFacilityName());
			assertNotNull(dsFacility.getFacilityParentSiteCode());
			assertNotNull(dsFacility.getFacilityState());
			assertNotNull(dsFacility.getFacilityType());
			assertNotNull(dsFacility.getPatientICN());
			assertNotNull(dsFacility.getFacilityClinics());

			for (DSFacilityClinic dsFacilityClinic : dsFacility.getFacilityClinics()) {

				assertNotNull(dsFacilityClinic.getStopCode());
				assertNotNull(dsFacilityClinic.getProviders());

				for (DSFacilityProvider dsFacilityProvider : dsFacilityClinic.getProviders()) {

					assertNotNull(dsFacilityProvider.getFirstName());
					assertNotNull(dsFacilityProvider.getLastName());
					assertNotNull(dsFacilityProvider.getProviderClass());
					assertNotNull(dsFacilityProvider.getProviderId());
					assertNotNull(dsFacilityProvider.getProviderNamePrefix());
					assertNotNull(dsFacilityProvider.getProviderNameSuffix());
				}

			}

		}

	}


	private void setupMockPatientCorrespondIdentifiers( PatientIdentifier patientIdentifier ) {

		// 	[ICN#1008593069V747998, dfn-777#237, MH#1011214432V662527, EDIPI#P001]

		// Mock what should be returned from patientCorrelationGateway
		PatientIdentifiers mockCorrespondIdentifiers = new PatientIdentifiers( ) ;
		PatientIdentifier mockPatientIdentifierICN = new PatientIdentifier( ASSIGNING_AUTH_ICN , PATIENT_ID_ICN ) ;
		PatientIdentifier mockPatientIdentifierMH = new PatientIdentifier( ASSIGNING_AUTH_MH , PATIENT_ID_MH ) ;
		PatientIdentifier mockPatientIdentifierEDIPI = new PatientIdentifier( ASSIGNING_AUTH_EDIPI , PATIENT_ID_EDIPI ) ;
		PatientIdentifier mockPatientIdentifierDFN = new PatientIdentifier( ASSIGNING_AUTH_DFN , PATIENT_ID_DFN ) ;

		mockCorrespondIdentifiers.add(mockPatientIdentifierICN);
		mockCorrespondIdentifiers.add(mockPatientIdentifierMH);
		mockCorrespondIdentifiers.add(mockPatientIdentifierEDIPI);
		mockCorrespondIdentifiers.add(mockPatientIdentifierDFN);

		when(patientCorrelationGateway.getCorrespondIdentifiers(patientIdentifier)).thenReturn(mockCorrespondIdentifiers);
	}

	private void setupMockVarFacilityTeamMembers( ) {

		VARFacilityMembers facilityTeam = new VARFacilityMembers( ) ;
		facilityTeam.setFacilityMembers( generateFacilityTeamCollection( ) ) ;
		dsFacilityProviderDataService.setPatientRelationshipDaoTeamImpl(patientRelationshipDaoTeamImpl);

		Map<String, Object> inputParamsMap = new HashMap<String, Object>();
		inputParamsMap.put(DAOConstants.PATIENT_ICN, PATIENT_ID_ICN);
		inputParamsMap.put(DAOConstants.STOP_CODE_PAIRS, STOP_CODE_PAIRS);

		when(patientRelationshipDaoTeamImpl.getPatientData(inputParamsMap)).thenReturn(facilityTeam);
	}

	private void setupMockVarFacilityClinicMembers( ) {

		VARFacilityMembers facilityClinics = new VARFacilityMembers( ) ;
		facilityClinics.setFacilityMembers(generateFacilityClinicCollection( ) ) ;
		dsFacilityProviderDataService.setPatientRelationshipDaoClinicImpl(patientRelationshipDaoClinicImpl);

		Map<String, Object> inputParamsMap = new HashMap<String, Object>();
		inputParamsMap.put(DAOConstants.PATIENT_ICN, PATIENT_ID_ICN);
		inputParamsMap.put(DAOConstants.STOP_CODE_PAIRS, STOP_CODE_PAIRS);

		when(patientRelationshipDaoClinicImpl.getPatientData(inputParamsMap)).thenReturn(facilityClinics);
	}

	private Collection< FacilityMember> generateFacilityTeamCollection( ) {

		String[ ] teamNameAry = {  "TRI TEAM 07 *WH*",  "TRI TEAM 07 *WH*",  "TRI TEAM 07 *WH*",  "TRI TEAM 07 *WH*",  "TRI TEAM 07 *WH*",  "TRI TEAM 07 *WH*",  "TRI TEAM 07 *WH*",  "TRI TEAM 07 *WH*",  "TRI TEAM 07 *WH*",  "TRI TEAM 07 *WH*",  "PC PROVIDER 4 *WH*",  "PC PROVIDER 4 *WH*",  "PC PROVIDER 4 *WH*",  "PC PROVIDER 4 *WH*",  "PC PROVIDER 4 *WH*",  "PC PROVIDER 4 *WH*"};
		String[ ] possiblePrimaryAry = {  "Y",  "",  "",  "",  "",  "",  "",  "",  "",  "",  "Y",  "N",  "",  "N",  "",  ""};
		String[ ] staffNameAry = {  "Person306985001 ZZStaff306985001",  "Person133378001 ZZStaff133378001",  "Person291921001 ZZStaff291921001",  "Person290129001 ZZStaff290129001",  "Person4382001 ZZStaff4382001",  "Person217427001 ZZStaff217427001",  "Person347018001 ZZStaff347018001",  "Person7217001 ZZStaff7217001",  "Person212940001 ZZStaff212940001",  "Person76489001 ZZStaff76489001",  "Person80647001 ZZStaff80647001",  "Person217728001 ZZStaff217728001",  "Person173537001 ZZStaff173537001",  "Person33516001 ZZStaff33516001",  "Person291165001 ZZStaff291165001",  "Person300454001 ZZStaff300454001"};
		String[ ] teamPurposeAry = {  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE"};
		String[ ] providerPositionAry = {  "MD 7",  "LPN 7",  "NURSE  (RN) 7",  "DIETITIAN",  "SOCIAL WORKER",  "NURSE  (RN) 7A",  "NURSE  (RN) 7C",  "NURSE  (RN) 7B",  "NURSE  (RN) 7D",  "NURSE (RN)  7E",  "PROVIDER",  "SOCIAL WORKER",  "NURSE RN 2",  "CARE MANAGER",  "MSA",  "LPN2"};
		String[ ] providerSidAry = {  "9852128",  "478434",  "496174",  "488039",  "15301731",  "1203244858",  "459950",  "555542",  "706203",  "649617",  "90139",  "10161444",  "20488",  "1401714362",  "10839279",  "1404109355"};
		String[ ] teamSidAry = {  "1200003759",  "1200003759",  "1200003759",  "1200003759",  "1200003759",  "1200003759",  "1200003759",  "1200003759",  "1200003759",  "1200003759",  "1400000566",  "1400000566",  "1400000566",  "1400000566",  "1400000566",  "1400000566"};
		String[ ] facilityIdAry = {  "777",  "777",  "777",  "777",  "777",  "777",  "777",  "777",  "777",  "777",  "9999",  "9999",  "9999",  "9999",  "9999",  "9999"};
		String[ ] providerStandardPositionAry = {  "PHYSICIAN-PRIMARY CARE",  "NURSE (LPN)",  "CARE MANAGER",  "DIETITIAN",  "SOCIAL WORKER",  "NURSE (RN)",  "NURSE (RN)",  "NURSE (RN)",  "NURSE (RN)",  "NURSE (RN)",  "PHYSICIAN-PRIMARY CARE",  "SOCIAL WORKER",  "NURSE (RN)",  "CARE MANAGER",  "HEALTH TECHNICIAN",  "NURSE (LPN)"};

		Collection<FacilityMember> facilityMemberCollection = new ArrayList<FacilityMember>();

		for (int i = 0; i < teamNameAry.length ; i++) {

			FacilityMemberTeam fmt = new FacilityMemberTeam( ) ;
			fmt.setTeamName(teamNameAry[i]);
			fmt.setPossiblePrimary(possiblePrimaryAry[i]);
			fmt.setStaffName(staffNameAry[i]);
			fmt.setTeamPurpose(teamPurposeAry[i]);
			fmt.setProviderPosition(providerPositionAry[i]);
			fmt.setProviderSid(providerSidAry[i]);
			fmt.setTeamSid(teamSidAry[i]);
			fmt.setFacilityId(facilityIdAry[i]);
			fmt.setProviderStandardPosition(providerStandardPositionAry[i]);

			facilityMemberCollection.add(fmt);
		}

		return (facilityMemberCollection);

	}

	private Collection< FacilityMember > generateFacilityClinicCollection( ) {

		String[ ] institutionCodeAry = {  "777",  "9999",  "9999",  "9999"};
		String[ ] primaryStopCodeAry = {  "323",  "323",  "323",  "323"};
		String[ ] primaryStopCodeNameAry = {  "PRIMARY CARE/MEDICINE",  "PRIMARY CARE/MEDICINE",  "PRIMARY CARE/MEDICINE",  "PRIMARY CARE/MEDICINE"};
		String[ ] physicalLocationAry = {  "TRIDENT PRIMARY CARE",  "CA",  "VA BUTLER BLD#1",  "BLD#1 BASEMENT"};
		String[ ] institutionIENAry = {  "777",  "9999",  "",  ""};
		String[ ] teamSIDAry = {  "1200003759",  "1400000566",  "1400000566",  "1400000566"};
		String[ ] locationSIDAry = {  "1200014220",  "18755",  "1400040365",  "1400039008"};
		String[ ] facilityIdAry = {  "777",  "9999",  "9999",  "9999"};
		String[ ] teamPurposeAry = {  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE",  "PRIMARY CARE"};
		String[ ] teamNameAry = {  "TRI TEAM 07 *WH*",  "PC PROVIDER 4 *WH*",  "PC PROVIDER 4 *WH*",  "PC PROVIDER 4 *WH*"};
		String[ ] institutionSIDAry = {  "1",  "89877",  "-1",  "-1"};
		String[ ] locationNameAry = {  "TRI TEAM 01 RN",  "BUT OPT/GAMWO WH",  "BUT OPT/GAMWO SAT EH",  "BUT OPT/GAMWO LATE WH EH/General Medicine"};
		String[ ] locationTypeAry = {  "CLINIC",  "CLINIC",  "CLINIC",  "CLINIC"};
		String[ ] instituationNameAry = {  "TEST VAMC 3",  "VAMC LVDEV",  "",  "VAMC LVDEV"};
		String[ ] locationIENAry = {  "32",  "32",  "2296",  "23"};

		Collection<FacilityMember> facilityMemberCollection = new ArrayList<FacilityMember>();

		for (int i = 0; i < institutionCodeAry.length ; i++) {

			FacilityMemberClinic fmc = new FacilityMemberClinic( ) ;
			fmc.setInstitutionCode(institutionCodeAry[ i ]);
			fmc.setPrimaryStopCode(primaryStopCodeAry[ i]);
			fmc.setPrimaryStopCodeName(primaryStopCodeNameAry[ i]);
			fmc.setPhysicalLocation(physicalLocationAry[i]);
			fmc.setInstitutionIEN(institutionIENAry[i]);
			fmc.setTeamSID(teamSIDAry[i]);
			fmc.setLocationSID(locationSIDAry[i]);
			fmc.setFacilityId(facilityIdAry[i]);
			fmc.setTeamPurpose(teamPurposeAry[i]);
			fmc.setTeamName(teamNameAry[i]);
			fmc.setInstitutionSID(institutionSIDAry[i]);
			fmc.setLocationName(locationNameAry[i]);
			fmc.setLocationType(locationTypeAry[i]);
			fmc.setInstitutionName(instituationNameAry[i]);
			fmc.setLocationIEN(locationIENAry[i]);

			facilityMemberCollection.add(fmc);
		}

		return (facilityMemberCollection);
	}

	private Collection<FacilityMember> generateFacilityProviderCollection() {

		int facilityMemberCount = 4;
		String[] parentSiteCodeAry = { "442", "442", "442", "442" };
		String[] facilityTypeAry = { "CBOC", "M&ROC", "M&ROC", "CBOC" };
		String[] facilityCodeAry = { "442GD", "442", "442", "442GC" };
		String[] facilityNameAry = { "GREELEY CBOC", "CHEYENNE VAMC", "CHEYENNE VAMC", "FORT COLLINS" };
		String[] facilityCityAry = { "GREELEY", "CHEYENNE", "CHEYENNE", "FORT COLLINS" };
		String[] facilityAddr1Ary = { "2001 70TH AVE #200", "2360 EAST PERSHING BLVD", "2360 EAST PERSHING BLVD", "2509 RESEARCH BLVD" };
		String[] facilityAddr2Ary = { "", "", "", "" };
		String[] facilityStateAry = { "CO", "WY", "WY", "CO" };
		String[] providerSidAry = { "410732", "414680", "424748", "567670" };
		String[] providerClassAry = { "", "", "", "" };
		String[] providerFirstNameAry = { "Person2304", "Person2355", "Person2413", "Person3133" };
		String[] providerLastNameAry = { "ZZStaff2304", "ZZStaff2355", "ZZStaff2413", "ZZStaff3133" };
		String[] providerNamePrefixAry = { "Mr", "Mr", "Mr", "Mr" };
		String[] providerNameSuffixAry = { "", "", "", "" };
		String[] patientICNAry = { "1008593069V747998", "1008593069V747998", "1008593069V747998", "1008593069V747998" };
		String[] clinicStopCodeAry = { "502", "323", "323", "502" };

		Collection<FacilityMember> facilityMemberCollection = new ArrayList<FacilityMember>();

		for (int i = 0; i < facilityMemberCount; i++) {

			FacilityMemberProvider fmp = new FacilityMemberProvider();
			fmp.setFacilityParentSiteCode(parentSiteCodeAry[i]);
			fmp.setClinicStopCode(clinicStopCodeAry[i]);
			fmp.setFacilityCode(facilityCodeAry[i]);
			fmp.setFacilityName(facilityNameAry[i]);
			fmp.setFacilityCity(facilityCityAry[i]);
			fmp.setFacilityState(facilityStateAry[i]);
			fmp.setFacilityStreetAddress1(facilityAddr1Ary[i]);
			fmp.setFacilityStreetAddress2(facilityAddr2Ary[i]);
			fmp.setFacilityType(facilityTypeAry[i]);
			fmp.setProviderClass(providerClassAry[i]);
			fmp.setProviderFirstName(providerFirstNameAry[i]);
			fmp.setProviderLastName(providerLastNameAry[i]);
			fmp.setProviderNamePrefix(providerNamePrefixAry[i]);
			fmp.setProviderNameSuffix(providerNameSuffixAry[i]);
			fmp.setProviderSID(providerSidAry[i]);
			fmp.setPatientICN(patientICNAry[i]);

			facilityMemberCollection.add(fmp);
		}

		return (facilityMemberCollection);
	}

}
